$Messages = data {
#culture="en-US"
ConvertFrom-StringData -StringData @'
    notRunningSqlExpress = Not running Express edition of SQL Server. 
    notRunningDefaultInstance = Metasys did not install this instance of SQL server, so it is not affected by the security vulnerability.
    noChangesHaveBeenMade = No changes have been made.
    warningChange = This script will disable an unused SQL login.
    continuePrompt = Do you want to continue? (Yes/No)
    yes = Yes
    accountSuccess = The change has been successfully applied.
    accountFailure = The change was not successful.
    checkingFirewallRules = Checking firewall rules...
    firewallRulesMessage = The following firewall rules allow TCP port 1433 traffic to SQL Server:
    firewallRulesAction = If remote access to SQL Server is not needed, we recommend that you modify these rules to exclude or block port 1433.
    firewallRulesRemoteAccess = Remote access might be needed by applications which integrate directly with the Metasys Server database. An example of this would be the OpenBlue Bridge with Metasys SQL Connector. If remote access is needed, create a separate firewall rule which only allows inbound TCP port 1433 traffic from the machine that needs this access. 
    noFirewallRulesFound = No firewall rules were found which allow TCP port 1433 traffic to SQL Server, no firewall changes needed
'@
}

Import-LocalizedData -BindingVariable Messages -ErrorAction SilentlyContinue

function Test-SqlFirewallRules {
    Write-Host ""
    Write-Host $Messages.checkingFirewallRules
    $sqlport = 1433
    $firewallRules = Get-NetFirewallPortFilter | Where-Object { ($_.Protocol -eq 'TCP') -and ($_.LocalPort -eq $sqlport) } | Get-NetFirewallRule | Where-Object { ($_.Action -eq 'Allow') }
    if ($firewallRules){
        Write-Host $Messages.firewallRulesMessage
        $firewallRules | Select-Object -ExpandProperty DisplayName | ForEach-Object { "   $_" }
        Write-Host $Messages.firewallRulesAction
        Write-Host $Messages.firewallRulesRemoteAccess
    } else {
        Write-Host $Messages.noFirewallRulesFound
    }
}

function Invoke-LocalSql {
    param (
        [string]$Query,
        [switch]$AsQuery
    )

    # Connect to local SQL Server using the 'master' database and trust self-signed certificates
    $connectionString = "Server=localhost;Database=master;Integrated Security=True;TrustServerCertificate=True;"

    $connection = New-Object System.Data.SqlClient.SqlConnection $connectionString
    try {
        $connection.Open()
        $command = $connection.CreateCommand()
        $command.CommandText = $Query

        if ($AsQuery) {
            $reader = $command.ExecuteReader()
            $results = @()
            while ($reader.Read()) {
                $row = @{}
                for ($i = 0; $i -lt $reader.FieldCount; $i++) {
                    $row[$reader.GetName($i)] = $reader[$i]
                }
                $results += [PSCustomObject]$row
            }
            $reader.Close()
            return $results
        } else {
            $rows = $command.ExecuteNonQuery()            
        }
    }
    catch {
        Write-Error "$_"
    }
    finally {
        $connection.Close()
    }
}

function Disable-SaLogin {
    Invoke-LocalSql "ALTER LOGIN [sa] WITH PASSWORD = '$([guid]::NewGuid().ToString())'"
    Invoke-LocalSql "ALTER LOGIN [sa] DISABLE"
    
    $containsSa = Invoke-LocalSql "SELECT DISTINCT sl.loginname as 'Disabled Accounts' FROM syslogins sl JOIN sys.sql_logins sql ON sl.sid=sql.sid WHERE sql.is_disabled=1" -AsQuery |
        Where-Object { $_."Disabled Accounts" -eq "sa" }

    if ($containsSa) {
        Write-Host $Messages.accountSuccess
    } else {
        Write-Host $Messages.accountFailure
    }
}

function Test-DefaultInstanceOfSqlExpress {
    param (
        [string]$InstanceName = "MSSQLSERVER"
    )

    $service = Get-Service -Name $InstanceName -ErrorAction SilentlyContinue

    if (-not $service) {
        Write-Host $Messages.notRunningDefaultInstance
        return $false
    }

    $edition = Invoke-LocalSql "SELECT SERVERPROPERTY('EngineEdition') AS Edition" -AsQuery | Select-Object -ExpandProperty Edition
    if ($edition -ne 4) {
        Write-Host $Messages.notRunningSqlExpress
        return $false
    }

    return $true
}

if (Test-DefaultInstanceOfSqlExpress) {
    Write-Host $Messages.warningChange
    $userResponse = Read-Host $Messages.continuePrompt

    if ($userResponse -ne $Messages.yes) {
        Write-Host $Messages.noChangesHaveBeenMade
    } else {
        Disable-SaLogin
    }
}
else {
    Write-Host $Messages.noChangesHaveBeenMade
}

Test-SqlFirewallRules
